<?xml version="1.0" encoding="ISO-8859-1"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

	<xsl:output method="html" indent="yes" version="4.0" />

	<!-- Global variables -->
	<xsl:variable name="DICOM" select="/DICOM" />
	<!-- Language selection parameter -->
	<xsl:variable name="lang" select="$DICOM/@lang" />
	<xsl:variable name="currentTemplate" select="$DICOM/@templateBase" />
	<xsl:variable name="ImageDir" select="$DICOM/@SRImage" />
	<!-- Template root. -->
	<xsl:variable name="tp" select="document($currentTemplate)/Customizations" />
	<xsl:variable name="messagecatalog" select="document('../sr/MSGCAT.XML')/Messages" />

	<!-- Number of columns patient demographic fields are displayed across. 
		Default is 2. -->
	<xsl:variable name="demographicsColumns">
		<xsl:choose>
			<xsl:when test="$tp/Demographics/@columns != ''">
				<xsl:value-of select="$tp/Demographics/@columns" />
			</xsl:when>
			<xsl:otherwise>
				2
			</xsl:otherwise>
		</xsl:choose>
	</xsl:variable>
	<!-- Number of columns images are displayed across. Default is 2. -->
	<xsl:variable name="imgColumns">
		<xsl:choose>
			<xsl:when test="$tp/Images/@columns != ''">
				<xsl:value-of select="$tp/Images/@columns" />
			</xsl:when>
			<xsl:otherwise>
				2
			</xsl:otherwise>
		</xsl:choose>
	</xsl:variable>

	<!-- Notification of empty content content items with 1C values. -->
	<xsl:variable name="NO_CONTENT">
		-
	</xsl:variable>

	<!-- #################################################################################################### 
		Coding Scheme Designator string constants -->
	<!-- DICOM Controlled Terminology; PS 3.16 Content Mapping Resource, Annex 
		D -->
	<!-- SNOMED Version 3 -->
	<xsl:variable name="CSD_SNOMED">
		SNM3
	</xsl:variable>
	<!-- SNOMED-RT -->
	<xsl:variable name="CSD_SNOMED_RT">
		SRT
	</xsl:variable>
	<!-- GE Medical System internal -->
	<xsl:variable name="CSD_GEMS">
		99GEMS
	</xsl:variable>
	<xsl:variable name="LF">
		<xsl:value-of select="'&#x0A;'" />
	</xsl:variable>
	<xsl:variable name="NewLineSeparator">
		\n
	</xsl:variable>



	<xsl:template match="/">
		<html>
			<body style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<xsl:call-template name="StartParsing" />
			</body>
		</html>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: processSections Description: Process report and template sections. -->
	<xsl:template name="StartParsing">
		<xsl:for-each select="$tp/Sections/Section">
			<xsl:variable name="curSection" select="current()" />
			<xsl:variable name="curSectionPos" select="position()" />
			<xsl:choose>
				<!-- Header -->
				<xsl:when test="@type = 'REPORT_TITLE'">
					<xsl:call-template name="addHeaderSections" />
				</xsl:when>
				<!-- Patient information -->
				<xsl:when test="@type = 'PATIENT_DEMOGRAPHICS'">
					<xsl:call-template name="addPDSections">
						<xsl:with-param name="pPDSection" select="$curSection" />
						<xsl:with-param name="pPosition" select="$curSectionPos" />
					</xsl:call-template>
				</xsl:when>
				<!-- SR content in the original order -->
				<xsl:when test="@name = 'LBL_SRSECTIONS'">
					<!-- Relationship type equals HAS OBS CONTEXT and Observation context -->
					<xsl:if
						test="$DICOM/ContentSequence/Item[RelationshipType = 'HAS OBS CONTEXT']">
						<xsl:call-template name="addObservationContext" />
					</xsl:if>
					<xsl:call-template name="processReportSections">
						<xsl:with-param name="pCurSectionPos" select="$curSectionPos" />
					</xsl:call-template>
				</xsl:when>
				<xsl:otherwise>

				</xsl:otherwise>
			</xsl:choose>
		</xsl:for-each>
	</xsl:template>
	
	<!-- #################################################################################################### 
		Template: addHeaderSections Description: Add header sections to the report 
		in the order specified by template. Parameters: pSection : Current Section 
		node with @type = 'HEADER' pPosition : Position of pSection -->
	<xsl:template name="addHeaderSections">
		<h1 style="text-align: center; font-weight: bold;">
			<xsl:value-of select="$DICOM/ConceptNameCodeSequence/Item/CodeMeaning" />
		</h1>
		<br></br>
		<br></br>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addPDSections Description: Add patient information to the report 
		in the order specified by template. -->
	<xsl:template name="addPDSections">
		<xsl:param name="pPDSection" />
		<xsl:param name="pPosition" />

		<xsl:for-each select="$pPDSection/Section">
			<xsl:choose>
				<!-- Identification -->
				<xsl:when
					test="@type = 'IDENTIFICATION' and count($tp/Demographics/Identification/Item[@add = 'true']) &gt; 0">
					<xsl:call-template name="addPDIdentification" />
				</xsl:when>
				<!-- Physician Info -->
				<xsl:when
					test="@type = 'PHYSICIAN_INFO' and count($tp/Demographics/PhysicianInfo/Item[@add = 'true']) &gt; 0">
					<xsl:call-template name="addPDPhysicianInfo" />
				</xsl:when>
			</xsl:choose>
		</xsl:for-each>
		<xsl:call-template name="addCompletionVerification" />
	</xsl:template>

	<!-- #################################################################################################### 
		Template: processReportSections Description: Process report sections. Parameters: 
		pCurSectionPos : -->
	<xsl:template name="processReportSections">
		<xsl:param name="pCurSectionPos" />
		<h1 style="text-align: left; font-weight: bold; color: #FF0000;">Images are not clinically applicable !</h1>
		<xsl:for-each
			select="$DICOM/ContentSequence/Item[RelationshipType = 'CONTAINS']">
			<xsl:call-template name="addContentItem">
				<xsl:with-param name="pItem" select="current()" />
				<xsl:with-param name="pPosition"
					select="concat($pCurSectionPos, position())" />
			</xsl:call-template>
		</xsl:for-each>
	</xsl:template>


	<!-- #################################################################################################### 
		Template: addPDemographics Description: Add patient demographics. pPDItems 
		: Patient demographics items in the template. pN : # of columns patient demographics 
		are displayed across. -->
	<xsl:template name="addPDemographics">
		<xsl:param name="pPDItems" />
		<xsl:param name="pN" select="$demographicsColumns" />
		<xsl:if test="$pPDItems">
			<table
				style="width: 100%; font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<xsl:call-template name="addNPDemographics">
					<xsl:with-param name="pPDItems" select="$pPDItems" />
					<xsl:with-param name="pN" select="$pN" />
				</xsl:call-template>
			</table>
			<hr />
		</xsl:if>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addNPDemographics Description: Recursively adds patient demographics 
		organized into N columns. Parameters: pPDItems : Patient demographics items 
		in the template. pN : # of columns patient demographics are displayed across. -->
	<xsl:template name="addNPDemographics">
		<xsl:param name="pPDItems" />
		<xsl:param name="pN" select="$demographicsColumns" />

		<xsl:variable name="firstNPDFields" select="$pPDItems[position() &lt;= $pN]" />
		<tr>
			<xsl:for-each select="$firstNPDFields">
				<td style="width: 20%; font-weight: bold">
					<xsl:variable name="fieldMsg"
						select="$messagecatalog/*[local-name() = current()/@name]" />
					<xsl:value-of select="$fieldMsg/msg[@lang = $lang]" />
					:
				</td>
				<td style="width: 30%;">
					<xsl:text> </xsl:text>
					<xsl:call-template name="addPDValue">
						<xsl:with-param name="pPDItem" select="current()" />
					</xsl:call-template>
				</td>
			</xsl:for-each>
		</tr>
		<xsl:variable name="remainingPDFields" select="$pPDItems[position() &gt; $pN]" />
		<xsl:if test="count($remainingPDFields) != 0">
			<xsl:call-template name="addNPDemographics">
				<xsl:with-param name="pPDItems" select="$remainingPDFields" />
				<xsl:with-param name="pN" select="$pN" />
			</xsl:call-template>
		</xsl:if>
	</xsl:template>





	<!-- #################################################################################################### 
		Template: addPDIdentification Description: Add patient demographics: Identification. -->
	<xsl:template name="addPDIdentification">
		<xsl:call-template name="addPDemographics">
			<xsl:with-param name="pPDItems"
				select="$tp/Demographics/Identification/Item[@add = 'true' and @name != 'LBL_COMPLETION_FLAG' and @name != 'LBL_VERIFICATION_FLAG']" />
		</xsl:call-template>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addPDPhysicianInfo Description: Add patient demographics: Physician 
		Info. -->
	<xsl:template name="addPDPhysicianInfo">
		<xsl:variable name="tpPDPhysicianInfo" select="$tp/Demographics/PhysicianInfo" />
		<xsl:variable name="tpRadiologist"
			select="$tpPDPhysicianInfo/Item[@name = 'LBL_RADIOLOGIST']" />
		<xsl:variable name="tpOrdPhys"
			select="$tpPDPhysicianInfo/Item[@name = 'LBL_ORDERING_PHYSICIAN']" />
		<xsl:variable name="tpCardiologist"
			select="$tpPDPhysicianInfo/Item[@name = 'LBL_CARDIOLOGIST']" />
		<xsl:variable name="tpRefPhys"
			select="$tpPDPhysicianInfo/Item[@name = 'LBL_REFERRING_PHYSICIAN']" />

		<xsl:variable name="radiologist"
			select="$DICOM//Item[ConceptNameCodeSequence/Item/CodeValue = 'J-06170' and ConceptNameCodeSequence/Item/CodingSchemeDesignator = $CSD_SNOMED]" />
		<xsl:variable name="orderingPhysician"
			select="$DICOM//Item[ConceptNameCodeSequence/Item/CodeValue = 'J-0016E' and ConceptNameCodeSequence/Item/CodingSchemeDesignator = $CSD_SNOMED]" />
		<xsl:variable name="cardiologist"
			select="$DICOM//Item[ConceptNameCodeSequence/Item/CodeValue = 'J-0612B' and (ConceptNameCodeSequence/Item/CodingSchemeDesignator = $CSD_SNOMED or ConceptNameCodeSequence/Item/CodingSchemeDesignator = $CSD_SNOMED_RT)]" />
		<xsl:variable name="referringPhysician" select="$DICOM/ReferringPhysiciansName" />

		<xsl:if
			test="($radiologist and $tpRadiologist/@add = 'true') or ($orderingPhysician and $tpOrdPhys/@add = 'true') or ($cardiologist and $tpCardiologist/@add = 'true') or ($referringPhysician and $tpRefPhys/@add = 'true')">
			<table width="100%"
				style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<xsl:if
					test="($radiologist and $tpRadiologist/@add = 'true') or ($orderingPhysician and $tpOrdPhys/@add = 'true')">
					<tr>
						<!-- (J-06170, SNM3, Radiologist) -->
						<td style="width: 20%; font-weight: bold">
							<xsl:if test="$tpRadiologist/@add = 'true'">
								<xsl:value-of
									select="$messagecatalog/LBL_RADIOLOGIST/msg[@lang = $lang]" />
								:
							</xsl:if>
						</td>
						<td style="width: 30%;">
							<xsl:if test="$tpRadiologist/@add = 'true'">
								<xsl:call-template name="addValue">
									<xsl:with-param name="pCurrent" select="$radiologist" />
								</xsl:call-template>
							</xsl:if>
						</td>
						<!-- (J-0016E, SNM3, Doctor) -->
						<td style="width: 20%; font-weight: bold">
							<xsl:if test="$tpOrdPhys/@add = 'true'">
								<xsl:value-of
									select="$messagecatalog/LBL_ORDERING_PHYSICIAN/msg[@lang = $lang]" />
								:
							</xsl:if>
						</td>
						<td style="width: 30%;">
							<xsl:if test="$tpOrdPhys/@add = 'true'">
								<xsl:call-template name="addValue">
									<xsl:with-param name="pCurrent" select="$orderingPhysician" />
								</xsl:call-template>
							</xsl:if>
						</td>
					</tr>
				</xsl:if>
				<xsl:if
					test="($cardiologist and $tpCardiologist/@add = 'true') or ($referringPhysician and $tpRefPhys/@add = 'true')">
					<tr>
						<!-- (J-0612B, SNM3, Cardiologist) -->
						<td style="width: 20%; font-weight: bold">
							<xsl:if test="$tpCardiologist/@add = 'true'">
								<xsl:value-of
									select="$messagecatalog/LBL_CARDIOLOGIST/msg[@lang = $lang]" />
								:
							</xsl:if>
						</td>
						<td style="width: 30%;">
							<xsl:if test="$tpCardiologist/@add = 'true'">
								<xsl:call-template name="addValue">
									<xsl:with-param name="pCurrent" select="$cardiologist" />
								</xsl:call-template>
							</xsl:if>
						</td>
						<!-- (0008,0090): Referring Physician -->
						<td style="width: 20%; font-weight: bold">
							<xsl:if test="$tpRefPhys/@add = 'true'">
								<xsl:value-of
									select="$messagecatalog/LBL_REFERRING_PHYSICIAN/msg[@lang = $lang]" />
								:
							</xsl:if>
						</td>
						<td style="width: 30%;">
							<xsl:if test="$tpRefPhys/@add = 'true'">
								<xsl:value-of select="$referringPhysician" />
							</xsl:if>
						</td>
					</tr>
				</xsl:if>
			</table>
			<hr />
		</xsl:if>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addCompletionVerification Description: Add Completion Flag and 
		Verification Flag. -->
	<xsl:template name="addCompletionVerification">
		<xsl:variable name="tpPDIdentification" select="$tp/Demographics/Identification" />
		<xsl:variable name="tpCompFlag"
			select="$tpPDIdentification/Item[@name = 'LBL_COMPLETION_FLAG']" />
		<xsl:variable name="tpVerifFlag"
			select="$tpPDIdentification/Item[@name = 'LBL_VERIFICATION_FLAG']" />
		<xsl:if test="$tpCompFlag/@add = 'true' or $tpVerifFlag/@add = 'true'">
			<table width="100%"
				style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<tr>
					<!-- Completion Flag -->
					<td style="width: 20%; font-weight: bold">
						<xsl:if test="$tpCompFlag/@add = 'true'">
							<xsl:value-of
								select="$messagecatalog/LBL_COMPLETION_FLAG/msg[@lang = $lang]" />
							:
						</xsl:if>
					</td>
					<td style="width: 30%;">
						<xsl:if test="$tpCompFlag/@add = 'true'">
							<xsl:value-of select="$DICOM/CompletionFlag" />
						</xsl:if>
					</td>
					<!-- Verification Flag -->
					<td style="width: 20%; font-weight: bold">
						<xsl:if test="$tpVerifFlag/@add = 'true'">
							<xsl:value-of
								select="$messagecatalog/LBL_VERIFICATION_FLAG/msg[@lang = $lang]" />
							:
						</xsl:if>
					</td>
					<td style="width: 30%;">
						<xsl:if test="$tpVerifFlag/@add = 'true'">
							<xsl:value-of select="$DICOM/VerificationFlag" />
						</xsl:if>
					</td>
				</tr>
				<xsl:if
					test="$tpVerifFlag/@add = 'true' and $DICOM/VerificationFlag = 'VERIFIED'">
					<tr>
						<td style="width: 20%; font-weight: bold" />
						<td style="width: 30%;" />
						<td style="width: 20%; font-weight: bold">
							<xsl:value-of
								select="$messagecatalog/LBL_VERIFYING_OBSERVER/msg[@lang = $lang]" />
							:
						</td>
						<td style="width: 30%;">
							<xsl:value-of
								select="$DICOM/VerifyingObserverSequence/Item/VerifyingObserverName" />
						</td>
					</tr>
					<tr>
						<td style="width: 20%; font-weight: bold" />
						<td style="width: 30%;" />
						<td style="width: 20%; font-weight: bold">
							<xsl:value-of
								select="$messagecatalog/LBL_VERIFYING_ORGANIZATION/msg[@lang = $lang]" />
							:
						</td>
						<td style="width: 30%;">
							<xsl:value-of
								select="$DICOM/VerifyingObserverSequence/Item/VerifyingOrganization" />
						</td>
					</tr>
					<tr>
						<td style="width: 20%; font-weight: bold" />
						<td style="width: 30%;" />
						<td style="width: 20%; font-weight: bold">
							<xsl:value-of
								select="$messagecatalog/LBL_VERIFICATION_DATETIME/msg[@lang = $lang]" />
							:
						</td>
						<td>
							<xsl:value-of
								select="$DICOM/VerifyingObserverSequence/Item/VerificationDateTime" />
						</td>
					</tr>
				</xsl:if>
			</table>
			<hr />
		</xsl:if>
	</xsl:template>
	<!-- #################################################################################################### 
		Template: addObservationContext Description: Add Observation context to the 
		report. -->
	<xsl:template name="addObservationContext">
		<!--h2 style="font-family: {$tp/Font/@type}">Observation Context</h2 -->

		<!-- ==================================================================================================== 
			Observer context -->
		<xsl:if
			test="count($DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121005]) &gt; 0">
			<table width="100%" border="1"
				style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<caption style="font-size: larger">
					<b>Observer Context</b>
				</caption>
				<tr>
					<th>Observer Type</th>
					<th>Observer Identifying Attributes</th>
				</tr>

				<!-- Person observers -->
				<xsl:if
					test="count($DICOM/ContentSequence/Item[ConceptCodeSequence/Item/CodeValue = 121006]) &gt; 0">
					<tr>
						<td align="center">
							<xsl:value-of
								select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../CodeValue = '121006']" />
						</td>
						<td>
							<table width="100%"
								style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
								<tr>
									<th style="font-size: smaller">Name</th>
									<th style="font-size: smaller">Organization Name</th>
									<th style="font-size: smaller">Role in the Organization</th>
									<th style="font-size: smaller">Role in the Procedure</th>
								</tr>
								<!-- Process each Person Observer -->
								<xsl:for-each
									select="$DICOM/ContentSequence/Item[ConceptCodeSequence/Item/CodeValue = 121006]">
									<tr>
										<!-- Person Observer Name, Req: M -->
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="following-sibling::Item/PersonName[../ConceptNameCodeSequence/Item/CodeValue = 121008 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121006][1]) = generate-id(current())]" />
										</td>
										<!-- Person Observer's Organization Name, Req: U, Def: Institution 
											Name (0008,0080) -->
										<td align="center" style="font-size: smaller">
											<xsl:choose>
												<xsl:when
													test="following-sibling::Item[ConceptNameCodeSequence/Item/CodeValue = 121009 and generate-id(preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121006][1]) = generate-id(current())]">
													<xsl:value-of
														select="following-sibling::Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = 121009 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121006][1]) = generate-id(current())]" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$DICOM/InstitutionName" />
												</xsl:otherwise>
											</xsl:choose>
										</td>
										<!-- Person Observer's Role in the Organization, Req: U -->
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="following-sibling::Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = 121010 and generate-id(../../../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121006][1]) = generate-id(current())]" />
										</td>
										<!-- Person Observer's Role in the Procedure, Req: U -->
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="following-sibling::Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = 121011 and generate-id(../../../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121006][1]) = generate-id(current())]" />
										</td>
									</tr>
								</xsl:for-each>
							</table>
						</td>
					</tr>
				</xsl:if>

				<!-- Device observers -->
				<xsl:if
					test="count($DICOM/ContentSequence/Item[ConceptCodeSequence/Item/CodeValue = 121007]) &gt; 0">
					<tr>
						<td>
							<xsl:value-of
								select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../CodeValue = '121007']" />
						</td>
						<td>
							<table width="100%"
								style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
								<tr>
									<th style="font-size: smaller">Name</th>
									<th style="font-size: smaller">Manufacturer</th>
									<th style="font-size: smaller">Model Name</th>
									<th style="font-size: smaller">Serial Number</th>
								</tr>
								<!-- Process each Device Observer -->
								<xsl:for-each
									select="$DICOM/ContentSequence/Item[ConceptCodeSequence/Item/CodeValue = 121007]">
									<tr>
										<!-- Device Observer Name, Req: U, Def: Station Name (0008,1010) -->
										<td align="center" style="font-size: smaller">
											<xsl:choose>
												<xsl:when
													test="following-sibling::Item[ConceptNameCodeSequence/Item/CodeValue = 121013 and generate-id(preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]">
													<xsl:value-of
														select="following-sibling::Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = 121013 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$DICOM/StationName" />
												</xsl:otherwise>
											</xsl:choose>
										</td>
										<!-- Device Observer Manufacturer, Req: U, Def: Manufacturer (0008,0070) -->
										<td align="center" style="font-size: smaller">
											<xsl:choose>
												<xsl:when
													test="following-sibling::Item[ConceptNameCodeSequence/Item/CodeValue = 121014 and generate-id(preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]">
													<xsl:value-of
														select="following-sibling::Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = 121014 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$DICOM/Manufacturer" />
												</xsl:otherwise>
											</xsl:choose>
										</td>
										<!-- Device Observer Model Name, Req: U, Def: Manufacturer's Model 
											Name (0008,1090) -->
										<td align="center" style="font-size: smaller">
											<xsl:choose>
												<xsl:when
													test="following-sibling::Item[ConceptNameCodeSequence/Item/CodeValue = 121015 and generate-id(preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]">
													<xsl:value-of
														select="following-sibling::Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = 121015 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$DICOM/ManufacturersModelName" />
												</xsl:otherwise>
											</xsl:choose>
										</td>
										<!-- Device Observer Serial Number, Req: U, Def: Device Serial 
											Number (0018,1000) -->
										<td align="center" style="font-size: smaller">
											<xsl:choose>
												<xsl:when
													test="following-sibling::Item[ConceptNameCodeSequence/Item/CodeValue = 121016 and generate-id(preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]">
													<xsl:value-of
														select="following-sibling::Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = 121016 and generate-id(../preceding-sibling::Item[ConceptCodeSequence/Item/CodeValue = 121007][1]) = generate-id(current())]" />
												</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$DICOM/DeviceSerialNumber" />
												</xsl:otherwise>
											</xsl:choose>
										</td>
									</tr>
								</xsl:for-each>
							</table>
						</td>
					</tr>
				</xsl:if>
			</table>
			<br />
		</xsl:if>

		<!-- ==================================================================================================== 
			Procedure context -->
		<xsl:if
			test="count($DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue &gt;= 121018 and ConceptNameCodeSequence/Item/CodeValue &lt;= 121023]) &gt; 0">
			<table width="100%" border="1"
				style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<caption style="font-size: larger">
					<b>Procedure Context</b>
				</caption>
				<tr>
					<th>Study Instance UID</th>
					<th>Accession Number</th>
				</tr>
				<tr>
					<!-- Procedure Study Instance UID, Req: U, Def: StudyInstanceUID (0020,000d) -->
					<td align="center">
						<xsl:choose>
							<xsl:when
								test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121018]">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/UID[../ConceptNameCodeSequence/Item/CodeValue = '121018']" />
							</xsl:when>
							<xsl:otherwise>
								<xsl:value-of select="$DICOM/StudyInstanceUID" />
							</xsl:otherwise>
						</xsl:choose>
					</td>
					<!-- Procedure Accession Number, Req: U, Def: Accession Number (0008,0050) -->
					<td align="center">
						<xsl:choose>
							<xsl:when
								test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121022]">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121022']" />
							</xsl:when>
							<xsl:otherwise>
								<xsl:value-of select="$DICOM/AccessionNumber" />
							</xsl:otherwise>
						</xsl:choose>
					</td>
				</tr>
			</table>
			<br />
		</xsl:if>

		<!-- ==================================================================================================== 
			Subject context -->
		<xsl:if
			test="count($DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121024]) &gt; 0">
			<table width="100%" border="1"
				style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
				<caption style="font-size: larger">
					<b>
						Subject Context:
						<xsl:value-of
							select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121024']" />
					</b>
				</caption>
				<xsl:choose>
					<!-- Patient subject -->
					<xsl:when
						test="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121024'] = '121025'">
						<tr>
							<th>Name</th>
							<th>ID</th>
							<th>Birth Date</th>
							<th>Sex</th>
							<th>Age</th>
							<th>Species</th>
						</tr>
						<tr>
							<!-- Subject Name, Req: MC, Def: Patient Name (0010,0010) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121029]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/PersonName[../ConceptNameCodeSequence/Item/CodeValue = '121029']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/PatientName" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Subject ID, Req: MC, Def: Patient ID (0010,0020) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121030]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121030']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/PatientID" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Subject Birth Date, Req: U, Def: Patient Birth Date (0010,0030) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121031]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/Date[../ConceptNameCodeSequence/Item/CodeValue = '121031']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/PatientBirthDate" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Subject Sex, Req: U, Def: Patient Sex (0010,0040) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121032]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121032']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/PatientSex" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Subject Age, Req: U, Def: Patient Age (0010,1010) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121033]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/MeasuredValueSequence/Item/NumericValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121033']" />
										<xsl:value-of
											select="concat(' ', $DICOM/ContentSequence/Item/MeasuredValueSequence/Item/MeasurementUnitsCodeSequence/Item/CodeMeaning[../../../../../ConceptNameCodeSequence/Item/CodeValue = '121033'])" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/PatientAge" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Subject Species, Req: MC, Def: (L-85B00, SNM3, homo sapiens) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121034]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121034']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:text>homo sapiens</xsl:text>
									</xsl:otherwise>
								</xsl:choose>
							</td>
						</tr>
					</xsl:when>
					<!-- Fetus subject -->
					<xsl:when
						test="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121024'] = '121026'">
						<tr>
							<th>Mother</th>
							<th>ID</th>
							<th>Number</th>
							<th>Number of Fetuses</th>
						</tr>
						<tr>
							<td align="center">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/PersonName[../ConceptNameCodeSequence/Item/CodeValue = '121036']" />
							</td>
							<td align="center">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121030']" />
							</td>
							<td align="center">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/MeasuredValueSequence/Item/NumericValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121037']" />
							</td>
							<td align="center">
								<xsl:value-of
									select="$DICOM/ContentSequence/Item/MeasuredValueSequence/Item/NumericValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121038']" />
							</td>
						</tr>
					</xsl:when>
					<!-- Specimen subject -->
					<xsl:when
						test="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121024'] = '121027'">
						<tr>
							<th>Accession Number</th>
							<th>Patient Context</th>
							<th>ID</th>
							<th>Type</th>
						</tr>
						<tr>
							<!-- Specimen Accession Number, Req: U, Def: Specimen Accession Number 
								(0040,050a) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121040]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121040']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$DICOM/SpecimenAccessionNumber" />
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Patient Subject Context -->
							<td align="center">
								<table width="100%"
									style="font-family: {$tp/Font/@type}; font-size: {$tp/Font/@size}pt">
									<tr>
										<th style="font-size: smaller">Name</th>
										<th style="font-size: smaller">ID</th>
										<th style="font-size: smaller">Birth Date</th>
										<th style="font-size: smaller">Sex</th>
										<th style="font-size: smaller">Age</th>
										<th style="font-size: smaller">Species</th>
									</tr>
									<tr>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/PersonName[../ConceptNameCodeSequence/Item/CodeValue = '121029']" />
										</td>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121030']" />
										</td>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/Date[../ConceptNameCodeSequence/Item/CodeValue = '121031']" />
										</td>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121032']" />
										</td>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/MeasuredValueSequence/Item/NumericValue[../../../ConceptNameCodeSequence/Item/CodeValue = '121033']" />
										</td>
										<td align="center" style="font-size: smaller">
											<xsl:value-of
												select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121034']" />
										</td>
									</tr>
								</table>
							</td>
							<!-- Specimen Identifier, Req: U, Def: Specimen Identifier (0040,0551) 
								if a single item is present in Specimen Sequence (0040,0550) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121041]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/TextValue[../ConceptNameCodeSequence/Item/CodeValue = '121041']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:if test="count($DICOM/SpecimenSequence/Item) = 1">
											<xsl:value-of
												select="$DICOM/SpecimenSequence/Item/SpecimenIdentifier" />
										</xsl:if>
									</xsl:otherwise>
								</xsl:choose>
							</td>
							<!-- Specimen Type, Req: U, Def: Specimen Type Code Sequence (0040,059a) 
								value if a single item is present in Specimen Sequence (0040,0550) -->
							<td align="center">
								<xsl:choose>
									<xsl:when
										test="$DICOM/ContentSequence/Item[ConceptNameCodeSequence/Item/CodeValue = 121042]">
										<xsl:value-of
											select="$DICOM/ContentSequence/Item/ConceptCodeSequence/Item/CodeMeaning[../../../ConceptNameCodeSequence/Item/CodeValue = '121042']" />
									</xsl:when>
									<xsl:otherwise>
										<xsl:if test="count($DICOM/SpecimenSequence/Item) = 1">
											<xsl:value-of
												select="$DICOM/SpecimenSequence/Item/SpecimenTypeCodeSequence/Item/CodeMeaning" />
										</xsl:if>
									</xsl:otherwise>
								</xsl:choose>
							</td>
						</tr>
					</xsl:when>
				</xsl:choose>
			</table>
		</xsl:if>
	</xsl:template>
	<!-- #################################################################################################### 
		Template: addContentItem Description: Add content item to report. Parameters: 
		pItem : First level ContentItem in the report. -->
	<xsl:template name="addContentItem">
		<xsl:param name="pItem" />
		<xsl:param name="pPosition" />

		<a
			name="{concat(translate($pItem/ConceptNameCodeSequence/Item/CodeMeaning, ' ', ''), $pPosition)}" />
		<xsl:if test="$pItem/ConceptNameCodeSequence/Item/CodeMeaning">
			<h2>
				<xsl:value-of select="$pItem/ConceptNameCodeSequence/Item/CodeMeaning" />
			</h2>
		</xsl:if>
		<xsl:call-template name="addItemWithGeneric">
			<xsl:with-param name="pItem" select="$pItem" />
		</xsl:call-template>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addPDValue Description: Add patient demographics value specific 
		to the field. pPDItem : Patient demographics item in the template. -->
	<xsl:template name="addPDValue">
		<xsl:param name="pPDItem" />

		<xsl:choose>
			<!-- Identification -->
			<!-- Patient ID -->
			<xsl:when test="$pPDItem/@name = 'LBL_PATIENTID'">
				<xsl:value-of select="$DICOM/PatientID" />
			</xsl:when>
			<!-- Patient Name -->
			<xsl:when test="$pPDItem/@name = 'LBL_NAME'">
				<xsl:value-of select="$DICOM/PatientName" />
			</xsl:when>
			<!-- Birthdate -->
			<xsl:when test="$pPDItem/@name = 'LBL_BIRTHDATE'">
				<xsl:value-of select="$DICOM/PatientBirthDate" />
			</xsl:when>
			<!-- Age -->
			<xsl:when test="$pPDItem/@name = 'LBL_AGE'">
				<xsl:value-of select="$DICOM/PatientAge" />
			</xsl:when>
			<!-- Sex -->
			<xsl:when test="$pPDItem/@name = 'LBL_SEX'">
				<xsl:value-of select="$DICOM/PatientSex" />
			</xsl:when>
			<!-- Study date and time -->
			<xsl:when test="$pPDItem/@name = 'LBL_STUDYDATE'">
				<xsl:value-of select="$DICOM/StudyDate" />
				<xsl:text>, </xsl:text>
				<xsl:value-of select="$DICOM/StudyTime" />
			</xsl:when>
			<!-- Study ID -->
			<xsl:when test="$pPDItem/@name = 'LBL_STUDYID'">
				<xsl:value-of select="$DICOM/StudyID" />
			</xsl:when>
			<!-- Accession Number -->
			<xsl:when test="$pPDItem/@name = 'LBL_ACCESSION_NUMBER'">
				<xsl:value-of select="$DICOM/AccessionNumber" />
			</xsl:when>
			<!-- Content date and time -->
			<xsl:when test="$pPDItem/@name = 'LBL_CONTENT_DATE'">
				<xsl:value-of select="$DICOM/ContentDate" />
				<xsl:text>, </xsl:text>
				<xsl:value-of select="$DICOM/ContentTime" />
			</xsl:when>

			<!-- ======================================== Medical History ======================================== -->
			<!-- Weight -->
			<xsl:when test="$pPDItem/@name = 'LBL_WEIGHT'">
				<xsl:value-of select="$DICOM/PatientWeight" />
			</xsl:when>
			<!-- Height -->
			<xsl:when test="$pPDItem/@name = 'LBL_HEIGHT'">
				<xsl:value-of select="$DICOM/PatientSize" />
			</xsl:when>

			<!-- ======================================== Physician Info ======================================== -->
			<!-- (J-0016E, SNM3, Doctor) -->
			<!--xsl:when test="$pPDItem/@name = 'LBL_ORDERING_PHYSICIAN'"> <xsl:variable 
				name="orderingPhysician" select="$DICOM//Item[ConceptNameCodeSequence/Item/CodeValue 
				= 'J-0016E' and ConceptNameCodeSequence/Item/CodingSchemeDesignator = $CSD_SNOMED]"/> 
				<xsl:call-template name="addValue"> <xsl:with-param name="pCurrent" select="$orderingPhysician"/> 
				</xsl:call-template> </xsl:when -->
			<!-- (J-06170, SNM3, Radiologist) -->
			<xsl:when test="$pPDItem/@name = 'LBL_REFERRING_PHYSICIAN'">
				<xsl:variable name="referringPhysician" select="$DICOM/ReferringPhysiciansName" />
				<xsl:value-of select="$referringPhysician" />
			</xsl:when>

			<!-- Completion Flag -->
			<xsl:when test="$pPDItem/@name = 'LBL_COMPLETION_FLAG'">
				<xsl:value-of select="$DICOM/CompletionFlag" />
			</xsl:when>
			<!-- Verification Flag -->
			<xsl:when test="$pPDItem/@name = 'LBL_VERIFICATION_FLAG'">
				<xsl:value-of select="$DICOM/VerificationFlag" />
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$DICOM/*[@tag = $pPDItem/@tag]" />
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- #################################################################################################### 
		Template: addValue Description: Add the value of content item depending on 
		the value type to HTML. Parameters: pCurrent : The current node of processing 
		pUnit : Enumerated values: cv: Code Value, cm: Code Meaning of Measurement 
		Units Code Sequence, none: don't add unit. -->

	<xsl:template name="addValue">
		<xsl:param name="pCurrent" />
		<xsl:param name="pUnit" select="'cv'" />
		<xsl:choose>
			<xsl:when test="$pCurrent/ValueType_TEXT or $pCurrent/ValueType = 'TEXT'">
				<xsl:choose>
					<xsl:when test="contains($pCurrent/TextValue, $LF)">
						<xsl:call-template name="break2NewLine">
							<xsl:with-param name="pText" select="$pCurrent/TextValue" />
							<xsl:with-param name="pNewLineSeparator" select="$LF" />
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="$pCurrent/TextValue" />
					</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<xsl:when test="$pCurrent/ValueType_CODE or $pCurrent/ValueType = 'CODE'">
				<xsl:call-template name="addCODEValue">
					<xsl:with-param name="pCurrent" select="$pCurrent" />
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="$pCurrent/ValueType_NUM or $pCurrent/ValueType = 'NUM'">
				<xsl:value-of select="$pCurrent/MeasuredValueSequence/Item/NumericValue" />
				<xsl:choose>
					<xsl:when test="$pUnit = 'cv'">
						<xsl:if
							test="$pCurrent/MeasuredValueSequence/Item/MeasurementUnitsCodeSequence/Item/CodeValue != '1'">
							<xsl:value-of
								select="concat(' ', $pCurrent/MeasuredValueSequence/Item/MeasurementUnitsCodeSequence/Item/CodeValue)" />
						</xsl:if>
					</xsl:when>
					<xsl:when test="$pUnit = 'cm'">
						<xsl:value-of
							select="concat(' ', $pCurrent/MeasuredValueSequence/Item/MeasurementUnitsCodeSequence/Item/CodeMeaning)" />
					</xsl:when>
				</xsl:choose>
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_PNAME or $pCurrent/ValueType = 'PNAME'">
				<xsl:value-of select="$pCurrent/PersonName" />
			</xsl:when>
			<xsl:when test="$pCurrent/ValueType_DATE or $pCurrent/ValueType = 'DATE'">
				<xsl:value-of select="$pCurrent/Date" />
			</xsl:when>
			<xsl:when test="$pCurrent/ValueType_TIME or $pCurrent/ValueType = 'TIME'">
				<xsl:if test="$DICOM/@international_date_time = 'true'">
					<xsl:attribute name="title">HH:MM:SS</xsl:attribute>
				</xsl:if>
				<xsl:value-of select="$pCurrent/Time" />
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_DATETIME or $pCurrent/ValueType = 'DATETIME'">
				<xsl:if test="$DICOM/@international_date_time = 'true'">
					<xsl:attribute name="title">YYYY-MM-DD, HH:MM:SS</xsl:attribute>
				</xsl:if>
				<xsl:value-of select="$pCurrent/DateTime" />
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_UIDREF or $pCurrent/ValueType = 'UIDREF'">
				<xsl:value-of select="$pCurrent/UID" />
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_COMPOSITE or $pCurrent/ValueType = 'COMPOSITE'">
				<a href="{$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPInstanceUID}">
					<xsl:value-of
						select="$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPInstanceUID" />
				</a>
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_IMAGE or $pCurrent/ValueType = 'IMAGE'">
				<xsl:call-template name="addImage">
					<xsl:with-param name="pSOPClassUID"
						select="$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPClassUID" />
					<xsl:with-param name="pSOPInstUID"
						select="$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPInstanceUID" />
					<xsl:with-param name="pIconImageSeq"
						select="$pCurrent/ReferencedSOPSequence/Item/IconImageSequence" />
				</xsl:call-template>
			</xsl:when>
			<xsl:when
				test="$pCurrent/ValueType_WAVEFORM or $pCurrent/ValueType = 'WAVEFORM'">
				<xsl:message>
					<xsl:value-of select="$pCurrent/ValueType_WAVEFORM" />
				</xsl:message>
				<a href="{$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPInstanceUID}">
					<xsl:value-of
						select="$pCurrent/ReferencedSOPSequence/Item/ReferencedSOPInstanceUID" />
				</a>
			</xsl:when>
		</xsl:choose>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addCODEValue Description: Adds Code Meaning and "(CV, CSD, CM)" 
		in tooltip. Parameters: pCurrent : CODE content item -->
	<xsl:template name="addCODEValue">
		<xsl:param name="pCurrent" />
		<xsl:value-of select="$pCurrent/ConceptCodeSequence/Item/CodeMeaning" />
	</xsl:template>

	<xsl:template name="addImage">
		<xsl:param name="pSOPClassUID" />
		<xsl:param name="pSOPInstUID" />
		<xsl:param name="pIconImageSeq" />
		<a>
		<img width="256" height="256" name="temp"
				onclick="this.height=512;this.width=512" ondblclick="this.height=256;this.width=256">
		<xsl:attribute name="alt">
        <xsl:value-of
					select="$messagecatalog/SOPClasses/SOPClass[@uid = $pSOPClassUID]/@abbrev" />
        </xsl:attribute>
				<xsl:attribute name="src">
           <xsl:value-of select="concat($ImageDir, $pSOPInstUID, '.jpg')" />
    </xsl:attribute>
			</img>
		</a>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addItemWithGeneric Description: Add content item to report with 
		generic tree traversal. Parameters: pItem : Content item to add. -->
	<xsl:template name="addItemWithGeneric">
		<xsl:param name="pItem" />

		<!-- Add value of different Value Types except CONTAINER. -->
		<xsl:if test="not($pItem/ValueType_CONTAINER)">
			<xsl:call-template name="addValue">
				<xsl:with-param name="pCurrent" select="$pItem" />
			</xsl:call-template>
		</xsl:if>
		<!-- Process nested ContentSequences. -->
		<xsl:if test="$pItem/ContentSequence">
			<xsl:call-template name="addRecursiveContent">
				<xsl:with-param name="pLevel" select="0" />
				<xsl:with-param name="pCurrent" select="$pItem" />
			</xsl:call-template>
		</xsl:if>
	</xsl:template>

	<!-- #################################################################################################### 
		Template: addRecursiveContent Description: Add content in table rows with 
		recursion. Parameters: pLevel : Level of recursion pCurrent : The current 
		node of processing -->
	<xsl:template name="addRecursiveContent">
		<xsl:param name="pLevel" />
		<xsl:param name="pCurrent" />

		<table width="100%" border="0">
			<xsl:for-each select="$pCurrent/ContentSequence/Item">
				<tr>
					<td align="right">
						<xsl:variable name="fontSize">
							<xsl:choose>
								<!-- Empty if no font size is specififed in the template. -->
								<xsl:when test="$tp/Font/@size = ''" />
								<!-- Calculating font size based on template font size and recursion 
									level. -->
								<xsl:when test="($tp/Font/@size - ($pLevel * 2)) &gt; 6">
									<xsl:value-of select="$tp/Font/@size - ($pLevel * 2)" />
								</xsl:when>
								<!-- Smallest font is 6pt. -->
								<xsl:otherwise>
									6
								</xsl:otherwise>
							</xsl:choose>
						</xsl:variable>
						<table border="0" width="{100 - ($pLevel * 5)}%"
							style="font-family: {$tp/Font/@type}; font-size: {$fontSize}pt">
							<tr>
								<td style="width: 30%; vertical-align: top;">
									<!--xsl:if test="RelationshipType != 'CONTAINS'"> <i><xsl:value-of 
										select="RelationshipType"/>:</i> <xsl:text> </xsl:text> </xsl:if -->
									<b>
										<xsl:value-of select="ConceptNameCodeSequence/Item/CodeMeaning" />
									</b>
								</td>
								<td style="vertical-align: top;">
									<xsl:call-template name="addValue">
										<xsl:with-param name="pCurrent" select="current()" />
									</xsl:call-template>
								</td>
							</tr>
						</table>
						<xsl:if test="current()/ContentSequence/Item">
							<xsl:call-template name="addRecursiveContent">
								<xsl:with-param name="pLevel" select="$pLevel + 1" />
								<xsl:with-param name="pCurrent" select="current()" />
							</xsl:call-template>
						</xsl:if>
					</td>
				</tr>
			</xsl:for-each>
		</table>

	</xsl:template>

	<!-- #################################################################################################### 
		Template: addImageCapture Description: Add one image optionally with name 
		and comment. Parameters: pImage : Image item. -->
	<xsl:template name="addImageCapture">
		<xsl:param name="pImage" />
		<xsl:param name="pWithName" select="'true'" />
		<xsl:param name="pWithComment" select="'true'" />

		<!-- Image -->
		<xsl:call-template name="addValue">
			<xsl:with-param name="pCurrent" select="$pImage" />
		</xsl:call-template>
	</xsl:template>

	<xsl:template name="addEmptyCells">
		<xsl:param name="pTotalCols" />
		<xsl:param name="pN" />

		<xsl:if test="$pN &gt; 0">
			<td style="width: {100 div $pTotalCols}%; text-align: center" />
			<xsl:if test="$pN &gt; 1">
				<xsl:call-template name="addEmptyCells">
					<xsl:with-param name="pTotalCols" select="$pTotalCols" />
					<xsl:with-param name="pN" select="$pN - 1" />
				</xsl:call-template>
			</xsl:if>
		</xsl:if>
	</xsl:template>
	<!-- #################################################################################################### 
		Template: break2NewLine Description: Break down $pNewLineSeparator separated 
		string and add <br/>. Parameters: pText : Text to break down. pNewLineSeparator 
		: new line character -->
	<xsl:template name="break2NewLine">
		<xsl:param name="pText" />
		<xsl:param name="pNewLineSeparator" select="$NewLineSeparator" />

		<xsl:choose>
			<xsl:when test="contains($pText, $pNewLineSeparator)">
				<xsl:value-of select="substring-before($pText, $pNewLineSeparator)" />
				<br />
				<xsl:call-template name="break2NewLine">
					<xsl:with-param name="pText"
						select="substring-after($pText, $pNewLineSeparator)" />
					<xsl:with-param name="pNewLineSeparator" select="$pNewLineSeparator" />
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$pText" />
				<br />
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
</xsl:stylesheet>
